import { NextRequest, NextResponse } from 'next/server';
import Database from 'better-sqlite3';
import path from 'path';

function getDb() {
    const dbPath = path.join(process.cwd(), 'comprehensive.db');
    return new Database(dbPath);
}

// POST /api/tasks/[id]/link-event - קישור משימה לאירוע
export async function POST(request: NextRequest, { params }: { params: Promise<{ id: string }> }) {
    const { id } = await params;
    const db = getDb();
    try {
        const body = await request.json();
        const { event_id } = body;

        if (!event_id) {
            return NextResponse.json(
                { error: 'מזהה אירוע הוא חובה' },
                { status: 400 }
            );
        }

        // בדיקה שהמשימה קיימת
        const task = db.prepare('SELECT id, customer_id FROM tasks WHERE id = ?').get(id) as unknown;
        if (!task) {
            return NextResponse.json({ error: 'משימה לא נמצאה' }, { status: 404 });
        }

        // בדיקה שהאירוע קיים
        const event = db.prepare('SELECT id, customer_id FROM events WHERE id = ?').get(event_id) as unknown;
        if (!event) {
            return NextResponse.json({ error: 'אירוע לא נמצא' }, { status: 404 });
        }

        // וולידציה: לוודא שהמשימה והאירוע שייכים לאותו לקוח
        if ((task as { customer_id: string }).customer_id !== (event as { customer_id: string }).customer_id) {
            return NextResponse.json(
                { error: 'המשימה והאירוע חייבים להיות שייכים לאותו לקוח' },
                { status: 400 }
            );
        }

        // עדכון המשימה עם event_id
        db.prepare('UPDATE tasks SET event_id = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ?').run(event_id, id);

        // קבלת המשימה המעודכנת
        const updatedTask = db.prepare(`
            SELECT 
                t.*,
                c.name as customer_name,
                c.company as customer_company,
                d.domain as domain_name,
                p.title as project_title,
                e.title as event_title,
                e.start_datetime as event_start
            FROM tasks t
            LEFT JOIN customers c ON t.customer_id = c.id
            LEFT JOIN domains d ON t.domain_id = d.id
            LEFT JOIN projects p ON t.project_id = p.id
            LEFT JOIN events e ON t.event_id = e.id
            WHERE t.id = ?
        `).get(id) as unknown;

        return NextResponse.json(updatedTask);

    } catch (e: unknown) {
        console.error('Link task to event error:', e);
        return NextResponse.json({ error: 'שגיאה בקישור משימה לאירוע: ' + (e as Error).message }, { status: 500 });
    } finally {
        db.close();
    }
}

// DELETE /api/tasks/[id]/link-event - ניתוק משימה מאירוע
export async function DELETE(request: NextRequest, { params }: { params: Promise<{ id: string }> }) {
    const { id } = await params;
    const db = getDb();
    try {
        // בדיקה שהמשימה קיימת
        const task = db.prepare('SELECT id FROM tasks WHERE id = ?').get(id) as unknown;
        if (!task) {
            return NextResponse.json({ error: 'משימה לא נמצאה' }, { status: 404 });
        }

        // ניתוק המשימה מהאירוע
        db.prepare('UPDATE tasks SET event_id = NULL, updated_at = CURRENT_TIMESTAMP WHERE id = ?').run(id);

        // קבלת המשימה המעודכנת
        const updatedTask = db.prepare(`
            SELECT 
                t.*,
                c.name as customer_name,
                c.company as customer_company,
                d.domain as domain_name,
                p.title as project_title
            FROM tasks t
            LEFT JOIN customers c ON t.customer_id = c.id
            LEFT JOIN domains d ON t.domain_id = d.id
            LEFT JOIN projects p ON t.project_id = p.id
            WHERE t.id = ?
        `).get(id) as unknown;

        return NextResponse.json(updatedTask);

    } catch (e: unknown) {
        console.error('Unlink task from event error:', e);
        return NextResponse.json({ error: 'שגיאה בניתוק משימה מאירוע: ' + (e as Error).message }, { status: 500 });
    } finally {
        db.close();
    }
}
